### A Pluto.jl notebook ###
# v0.20.4

using Markdown
using InteractiveUtils

# ╔═╡ 4bae382c-eb38-4b21-8b61-67d238c80d19
import Pkg; Pkg.activate(".")

# ╔═╡ 93f1b40e-8990-405d-bba1-9f98abd054ca
begin
	using CairoMakie
	using CommonMark
	using CSV
	using DataFrames
	using LinearAlgebra
	using LombScargle
	using PlutoUI
	using Random
end

# ╔═╡ ff837227-c0bc-433c-b462-9e18ff3d947b
md"""
**What is this?**


*This jupyter notebook is part of a collection of notebooks on various topics discussed during the Time Domain Astrophysics course delivered by Stefano Covino at the [Università dell'Insubria](https://www.uninsubria.eu/) in Como (Italy). Please direct questions and suggestions to [stefano.covino@inaf.it](mailto:stefano.covino@inaf.it).*
"""

# ╔═╡ 53a01399-1f0b-4fbb-9966-97124e9eff42
md"""
**This is a `Julia` notebook**
"""

# ╔═╡ 414e8cd2-d282-4920-b4e8-91ed552467cf
Pkg.instantiate()

# ╔═╡ f7100fe5-2cfd-448d-abcc-66a623a8f11f
# ╠═╡ show_logs = false
md"""
$(LocalResource("Pics/TimeDomainBanner.jpg"))
"""

# ╔═╡ 31dd4ceb-8afd-4b55-aacb-2d888681ad1c
# ╠═╡ show_logs = false
md"""
# Unevenly sampled data

***

- Harmonic analysis of unevenly spaced data is problematic due to the loss of information and increase in aliasing.

- A (partial) solution to this problem was proposed in astronomy and it is now widely used and known as *Lomb-Scargle periodogram*.

- In case of regular sampling it is equivalent to the Fourier periodogram.

- An example of LS periodogram compute analying optical observation of the LINEAR object [11375941](https://simbad.u-strasbg.fr/simbad/sim-id?Ident=linear11375941&NbIdent=1&Radius=2&Radius.unit=arcmin&submit=submit+id) is shown below:

$(LocalResource("Pics/linear11375941lc.png"))
$(LocalResource("Pics/linear11375941ls.png"))
"""

# ╔═╡ e6386bbc-ea18-49a6-b64a-51b627d31565
md"""
## Many possible techniques
***

- Fourier methods:
    - are based on the Fourier transform, power spectra and correlation functions (Schuster periodogram, Lomb-Scargle periodogram, correlation-based methods, wavelet transform, etc.)

- Phase-folding methods:
    - depend on folding observations as a function of phase (String Length, Analysis of Variance, Phase Dispersion Minimization, Gregory & Loredo method, conditional entropy method, correntropy methods, etc.)

- Least-squares methods:
    - involve fitting a model to the data at each candidate frequency (Lomb-Scargle periodogram, Supersmoother approach, orthogonal polynomial fits, etc.)

- Bayesian approaches:
    - apply Bayesian probability theory to the problem (Lomb-Scargle generalisation, Gregory & Loredo, Gaussian process models, stochastic process models, etc.)

"""

# ╔═╡ 477704cc-f022-4cc5-a4a4-679387b819bd
# ╠═╡ show_logs = false
md"""
## A reminder of the CFT
***

- Given a continuous signal g(t) the Fourier transform and its inverse are:

```math
\hat{g}(f) \equiv \int_{-\infty}^\infty g(t) e^{-2\pi i f t} dt \quad  g(t) \equiv \int_{-\infty}^\infty \hat{g}(f) e^{+2\pi i f t} df
```

- We can also define the Fourier transform operator:

```math
\mathcal{F}\{g\} = \hat{g} \quad    \mathcal{F}^{-1}\{\hat{g}\} = g
```

- g and ĝ are know as a Fourier pair: $g \Longleftrightarrow \hat{g}$.

- The Fourier Transform (FT) is a linear operator:

```math
\mathcal{F}\{f(t) + g(t)\} = \mathcal{F}\{f(t)\} + \mathcal{F}\{g(t)\}\nonumber \quad
       \mathcal{F}\{A f(t)\} = A\mathcal{F}\{f(t)\}
```

- The FT of a sinusoid with frequency $f_0$ is a sum of delta functions at $\pm f_0$, where $\delta(f)\equiv\int_{-\infty}^\infty e^{-2\pi i x f}df$.

- We can write: $ \mathcal{F}\{e^{2\pi f_0 t}\} = \delta(f - f_0)$, and:

```math
\mathcal{F}\{\cos(2\pi f_0 t)\} = \frac{1}{2}\left[\delta(f - f_0) + \delta(f + f_0)\right] \quad
      \mathcal{F}\{\sin(2\pi f_0 t)\} = \frac{1}{2i}\left[\delta(f - f_0) - \delta(f + f_0)\right]
```

- Relations that can be derived from Euler’s formula: $e^{ix} = \cos x + i\sin x$

- A time shift imparts a phase in the FT:  $  \mathcal{F}\{g(t - t_0)\} = \mathcal{F}\{g(t)\} e^{-2\pi i ft_0}$.

- And, as we know, the squared amplitude of the FT of a continuous signal is known as the power spectral density (PSD):  $ \mathcal{P}_g \equiv \left|\mathcal{F}\{g\}\right|^2 $.

    - Note that if $g$ is real-valued, it follows that $P_g$ is an even function, {i.e.} $\mathcal{P}_g(f) = \mathcal{P}_g(-f)$.

$(LocalResource("Pics/FTpairs.png"))
"""

# ╔═╡ f95ec17d-9c56-4897-b5b2-65c3470fc294
# ╠═╡ show_logs = false
md"""
### The convolution theorem
***

- A convolution of two functions, usually denoted by the $\ast$ symbol, is defined as follows: $[f \ast g](t) \equiv \int_{-\infty}^\infty f(\tau)g(t - \tau) d\tau$

$(LocalResource("Pics/convolution.png"))

- It can be shown that the FT of a convolution is the point-wise product of the individual FTs:

```math\mathcal{F}\{f \ast g\} = \mathcal{F}\{f\} \cdot \mathcal{F}\{g\} \qquad   \mathcal{F}\{f \cdot g\} = \mathcal{F}\{f\} \ast \mathcal{F}\{g\}
```

"""

# ╔═╡ 0f01200a-665e-4ae2-a6b9-b5c192955f3f
# ╠═╡ show_logs = false
cm"""
### Effect of the observing window on a FT
***

- If ``g_{obs}(t) = g(t)W(t)``, by the convolution theorem we have: ``\mathcal{F}\{g_{obs}\} = \mathcal{F}\{g\} \ast \mathcal{F}\{W\}``.

$(LocalResource("Pics/observingwindow.png"))

- If we assume to observe a continuous periodic signal over a limited span of time the observed signal is the pointwise product of the underlying infinite periodic signal with a rectangular window function.

- By the convolution theorem, the FT is given by the convolution of the transform of the underlying function (here a set of delta functions at the component frequencies) and the transform of the window function (here a sinc function).

- For the purely periodic signal  this convolution has the effect of replacing each delta function with a sinc function.

- Because of the inverse relationship between the width of the window and the width of its transform, it follows that a wider observing window leads to proportionally less spread in the FT of the observed function.

| $(LocalResource("Pics/densesampling.png")) | $(LocalResource("Pics/sparsesampling.png")) |
| ----------------------------------------- | ------------------------------------------- |

- The observed FT is a convolution of the true transform (here a localized Gaussian) and the window transform (here another Dirac comb). Howver, if the sampling rate is "too low", the result is that the FT of the window function has spacing narrower than the FT of the signal.

    - The observed FT suffers from aliasing of signals, such that not all frequency information can be recovered.

- This is the famous Nyquist sampling theorem. If we have a regularly-sampled function with a sampling rate of ``f_0 = 1/T``, we can only fully recover the frequency information if the signal is *band-limited* between frequencies ``\pm f_0/2``.
"""

# ╔═╡ a24629ad-7302-4aff-b7bf-fcc7635b9691
md"""
### The discrete Fourier transform (DFT)
***

- Given and infinitely long and continuous signal $g(t)$ observed on a regular grid with spacing $Δt$ we have:

```math
g_{obs} = g(t) III_{\Delta t}(t) \qquad \hat{g}_{obs}(f) = \sum_{n=-\infty}^\infty g(n\Delta t) e^{-2\pi i f n \Delta t}
```

- In the real world we never have an infinite number of observations, defining $g_n \equiv g(n\Delta t)$ we may write:

```math
\hat{g}_{obs}(f) = \sum_{n=0}^N g_n e^{-2\pi i f n \Delta t}
```

- or, with $\Delta f = 1 / (N\Delta t)$ and $\hat{g}_k \equiv \hat{g}_{obs}(k\Delta f)$:

```math
\hat{g}_k = \sum_{n=0}^N g_n e^{-2\pi i k n / N}
```

- which is the standard form of the DFT.
"""

# ╔═╡ 533e41e2-08c3-450b-83eb-ae19664a6cfe
md"""
### The Classical Periodogram
***

- Applying the definition of Fourier spectrum $P_g \equiv |F{g}|^2$ we then have:

```math
P_S(f) = \frac{1}{N}\left|\sum_{n=1}^N g_n e^{-2\pi i f t_n}\right|^2
```

- This is the periodogram, which is an “estimator” of the true power spectrum of the underlying continuum function $g(t)$. It is sometimes called the *Schuster periodogram*.

- This point should be well emphasized: the *periodogram* and the *power spectrum* are conceptually different things, i.e. the periodogram is the statistic we compute from our data, and it is an *estimator* of the power spectrum, the underlying continuous function of interest.

- As we know, it is not a consistent estimator since it suffers from intrinsic variance even in the limit of an infinite number of observations.

"""

# ╔═╡ ae7829b7-b652-4e54-9231-f7bc3c3b6093
# ╠═╡ show_logs = false
cm"""
### Nonuniform sampling
***

- In the real world, particularly in fields like Astronomy where observations are subject to influences of weather and diurnal, lunar, or seasonal cycles, the sampling rate is generally far from uniform.

- In the general non-uniform case, we measure some signal at a set of ``N`` times, denoted ``\{t_n\}``, and defining the following observing window:

```math
W_{\{t_n\}}(t) = \sum_{n=1}^{N} \delta(t - t_n)
```

- Applying this window to the true underlying signal ``g(t)``, we get:

```math
g_{obs}(t) = g(t) W_{\{t_n\}}(t) = \sum_{n=1}^{N} g(t_n)\delta(t - t_n)
```

- And, just as in the evenly-sampled case, the FT of the observed signal is a convolution of the transforms of the true signal and the window:

```math
\mathcal{F}\{g_{obs}\} = \mathcal{F}\{g\} \ast \mathcal{F}\{W_{\{t_n\}}\}
```

- Unlike in the uniform case, the window transform ``\mathcal{F}\{W_{\{t_n\}}\}`` will generally not be a straightforward sequence of delta functions, and the symmetry present in the Dirac comb is broken by the uneven sampling, leading the transform to be much more "noisy".

| $(LocalResource("Pics/sparseirregular.jpg", :height=>300)) | $(LocalResource("Pics/denseirregular.png", :height=>300)) |
| ------------------------------------------------------ | ---------------------------------------------------- |

- The irregular spacing within the observing window translates to irregular frequency peaks in its transform, causing the observed transform to be noisier. Even with very dense sampling of the function, the FT cannot be exactly recovered due to the imperfect aliasing present in the window transform.

- Eventualy, the FT of a non-uniformly spaced delta functions looks like random noise, and partly it is.
    - It reflects the “random” distribution of the sampling time.

- A denser sampling helps, but to some extent “noise” is unavoidable.

"""

# ╔═╡ 9a422815-783a-473c-8538-6ecf5c208bb0
# ╠═╡ show_logs = false
md"""
### However, (almost) no Nyquist limit!
***

- The Nyquist limit is a direct consequence of the symmetry in the Dirac comb window function that describes evenly-sampled data, and uneven sampling destroys the symmetry that underlies its definition.

- For unevenly-sampled data, a "Nyquist limit" might or might not exist, and even in cases where it does exist it tends to be far larger (and thus far less relevant) than in the evenly-sampled case.

$(LocalResource("Pics/nonyquist.jpg"))

- An example of data for which the various poorly-motivated "pseudo-Nyquist" approaches fail.
    - The upper panels show the data, a noisy sinusoid with a frequency of 100 (i.e. a period of 0.01).
    - The lower left panel shows a histogram of spacings between observations: the minimum spacing is 2.55, meaning that the signal has over 250 full cycles between the closest pair of observations.
- Nevertheless, the periodogram (lower right) clearly identifies the correct period, though it is orders of magnitude larger than pseudo-Nyquist estimates based on average or minimum sampling rate.
"""

# ╔═╡ 6fa31a81-cbbe-401b-acb0-0d44b0980363
cm"""
## The Lomb-Scargle periodogram
***

- The classical periodogram can be rewritten as:

```math
P(f) = \frac{1}{N}\left|\sum_{n=1}^N g_n e^{-2\pi i f t_n} \right|^2 = \frac{1}{N}\left[
    \left(\sum_n g_n \cos(2\pi f t_n)\right)^2
    + \left(\sum_n g_n \sin(2\pi f t_n)\right)^2
    \right]
```
    
- In principle this formula could be used for non-uniform sampling too, yet the obtained periodogram does not offer, in general, some of the useful statistical properties of the even sampling case.

- The problem was addressed by proposing a generalized expression:

```math
P(f) = \frac{A^2}{2}\left(\sum_n g_n \cos(2\pi f [t_n-\tau])\right)^2
       + \frac{B^2}{2} \left(\sum_n g_n \sin(2\pi f [t_n-\tau])\right)^2
``

- where ``A``, ``B``, and ``\tau`` are arbitrary functions of the frequency ``f`` and observing times ``\{t_i\}`` (but not the values ``\{g_n\}``).

- It is possible to define ``A``, ``B``, and ``\tau`` such as:
    - the periodogram reduces to the classical form in the case of equally spaced observations;
    - the periodogram distribution is analytically computable;
    - the periodogram is insensitive to global time shifts in the data.

- The values of $A$ and $B$ leading to these properties result in the following
form of the generalized periodogram:

```math
P_{LS}(f) =
  \frac{1}{2} \Bigg\{
  \bigg(\sum_n g_n \cos(2\pi f [t_n-\tau])\bigg)^2 \bigg/
  \sum_n \cos^2(2\pi f [t_n-\tau])  + ~ \bigg(\sum_n g_n \sin(2\pi f [t_n-\tau])\bigg)^2 \bigg/
  \sum_n \sin^2(2\pi f [t_n-\tau])  \Bigg\}
```

- where ``\tau`` is specified for each ``f`` to ensure time-shift invariance:

```math
\tau = \frac{1}{4\pi f}\tan^{-1}\Bigg(
  \frac{\sum_n \sin(4\pi f t_n)}{\sum_n \cos(4\pi f t_n)}\Bigg)
```

- This modified periodogram differs from the classical periodogram only to the extent that the denominators ``\sum_n \sin^2(2\pi f t_n)`` and ``\sum_n \cos^2(2\pi f t_n)`` differ from ``N/2``, which is the expected value of each of these quantities in the limit of complete phase sampling at each frequency.

- Thus, in many cases of interest the Lomb-Scargle periodogram only differs slightly (but not negligibly) from the classical/Schuster periodogram.

- An important result is that the LS periodogram could be obtained fitting a simple sinusoidal model to the data at each given frequency, and deriving the periodogram from the goodness of fit (``\chi^2``).

    - We will see later that this possible different derivation has important consequences.

"""

# ╔═╡ 367c08ff-cedd-4e63-b53a-3178b0ec45cb
md"""
#### Exercise about classical vs Lomb-Scargle periodograms
***

- Let's study a curve observed at 30 epochs with some noise. Although with a sparse sampling, it is a sinusoidal curve with period.
"""

# ╔═╡ ef919d43-ffb2-404f-94e4-a10b9a543e09
begin
	
	t = [48.68293613, 43.68819185, 95.71437369, 71.00603639, 48.54022079,
	       17.820341  , 89.19924435, 61.75940713, 95.14511764, 80.92812141,
	       68.32096035, 11.61911367, 76.40351048, 24.05532227, 91.73672731,
	       36.27323062, 39.10355853, 49.25845392, 93.18918459, 30.02253181,
	       29.31289813, 88.33416194, 94.6922681 , 59.80890946, 52.82566667,
	       90.27610636, 12.10326369,  4.5971768 , 68.08097096, 38.76998251]
	
	y = [-1.06749625, -0.25882508,  0.54888049,  0.50273813, -1.27428505,
	       -1.17429603,  0.79366548, -0.64418343,  0.59271595, -0.65160916,
	       -0.46201257, -0.88418838,  0.74365945, -1.13026806, -0.78967783,
	       -0.75395011,  0.72986805, -0.58586657, -0.60369567, -0.9487803 ,
	       -0.49721427,  1.21028735,  0.32603806,  0.0210877 ,  0.09798819,
	        0.78991859, -0.3939756 , -1.13497054, -1.0160708 ,  0.84690192]
	
	
	fg1 = Figure()
	
	ax1fg1 = Axis(fg1[1, 1])
	
	scatter!(t,y,)
	
	fg1
end

# ╔═╡ ce5b3710-ac14-44f5-85d8-abc9756c1f11
md"""
- Now, let's write one of the possible versions of the Fourier periodogram. This is not optimized as a FFT, yet it works for any input frequency set.
"""

# ╔═╡ b985cda2-8242-468b-b5de-2299bfa50b33
md"""
- And then compute the Lomb-Scargle periodogram and Schuster periodograms, together with the funcions characterized the Lomb-Scargle formula.
"""

# ╔═╡ 2e3e0747-48c2-43d1-abd4-b15f61a402bd
begin
	lsplan = LombScargle.plan(t,y,normalization=:psd,minimum_frequency=0,maximum_frequency=0.5,samples_per_peak=20)
	
	lsprg = lombscargle(lsplan)
end;

# ╔═╡ 18958f04-3eb0-4e4e-9957-ca0c5540e5a5
function SchusterPeriodogram(t, mag, freq)
    pwr = abs.(exp.(-2im .* π .* lsprg.freq * t') * y).^2
    return pwr/length(t)
end

# ╔═╡ ccb2f0e6-517b-4860-bac2-dac964e8d388
begin
	p_schuster = SchusterPeriodogram(t, y, lsprg.freq)
	
	tau = 1 ./ (4 .* π .* lsprg.freq[2:end]) .* atan.(sum(sin.(4 .* π .* lsprg.freq[2:end] * t'),dims=2), sum(cos.(4 .* π .* lsprg.freq[2:end] * t'),dims=2))
end;

# ╔═╡ 0b098634-7890-4551-8343-34dbba3159ba
begin
	sw = 0.
	cw = 0.
	for ep in t
	    global sw = sw .+ sin.(2 .* π .* lsprg.freq[2:end] .* (ep .- tau)).^2
	    global cw = cw .+ cos.(2 .* π .* lsprg.freq[2:end] .* (ep .- tau)).^2
	end
	sin_window = sw
	cos_window = cw
end;

# ╔═╡ 2503892b-3bcf-41b8-a636-562d2613bd59
md"""
- And, finally, let's plot the results.
"""

# ╔═╡ f652aeb7-bcf0-481c-924c-412f406b3542
begin
	fg2 = Figure(size=(1000,1000))
	
	ax1fg2 = Axis(fg2[1, 1],
	    title="Spectral Power"
	    )
	
	lines!(lsprg.freq[2:end],lsprg.power[2:end],label="Lomb-Scargle Periodogram",color=:blue)
	lines!(lsprg.freq[2:end],p_schuster[2:end],label="Classical Periodogram",color=:red)
	
	axislegend()
	
	xlims!(0,0.5)
	
	ax2fg2 = Axis(fg2[2, 1],
	    xlabel="Frequency"
	    )
	
	
	lines!(lsprg.freq[2:end],vec(cos_window),label=L"$\sum_n\ \cos^2 [2\pi f (t_n-\tau)]$",color=:red)
	lines!(lsprg.freq[2:end],vec(sin_window),label=L"$\sum_n\ \sin^2 [2\pi f (t_n-\tau)]$",color=:blue)
	hlines!(length(t)/2,linestyle=:dash,color=:black)
	
	axislegend()
	
	xlims!(0,0.5)
	ylims!(0,30)
	
	fg2
	
	
end

# ╔═╡ 637108d6-c977-4a2d-b312-e40bf375d5f1
md"""
- As anticipated, LS and Fourier periodogram are usually rather similar, and their differences rely in how much the quantities plotted in the bottom plot are different from N/2.
"""

# ╔═╡ 325a6b84-af05-4d78-94a2-3674a501c1c2
# ╠═╡ show_logs = false
md"""
## LS extensions
***

- Considering the LS formula as the result of a regular fitting procedure allows one to discussn interesting generalizations.

- In the least squares interpretation of the periodogram, a sinusoidal model is proposed at each candidate frequency $f$: $y(t;f) = A_f \sin(2 \pi f (t - \phi_f))$, where the amplitude $A_f$ and phase $\phi_f$ can vary as a function of frequency.

- These model parameters are fit to the data by constructing the $\chi^2$ statistic at each frequency: $\chi^2(f) \equiv \sum_n \big(y_n - y(t_n;f)\big)^2$ and minimizing $\chi^2(f)$ at with respect to $A_f$ and $\phi_f$.

- Denoting the minimum value as $\hat{\chi}^2(f)$ the LS periodogram can be equivalently written:

```math
P(f) = \frac{1}{2}\big[\hat{\chi}^2_0 - \hat{\chi}^2(f)\big]
```

- where $\hat{\chi}^2_0$ is the non-varying reference model.

- Basing on this view a trivial yet fundamental extension is to include in the fitting procedure the (Gaussian) errors on the data through the standard change to the $\chi^2$ expression:

```math
\chi^2(f) \equiv \sum_n \left(\frac{y_n - y_{model}(t_n;f)}{\sigma_n}\right)^2
```

- This generalization of $\chi^2(f)$ also suggests a convenient way to construct a periodogram in the presence of correlated observational noise. If we let $\Sigma$ denote the  $N\times N$ noise covariance matrix for $N$ observations:

```math
\vec{y} = [y_1, y_2,\cdots y_n]^T \qquad \vec{y}_{model} = [y_{model}(t_1),y_{model}(t_1),\cdots y_{model}(t_n)]^T
```

- and therefore:

```math
\chi^2(f) = (\vec{y}-\vec{y}_{model})^T\Sigma^{-1}(\vec{y}-\vec{y}_{model})
```

- which reduces to the previous form if noise is uncorrelated (i.e., if the off-diagonal terms of $\Sigma$ are zero).

- Another import extension involves adding an offset term to the sinusoidal model at each frequency (the "floating mean" model):

```math
y_{model}(t;f) = y_0(f) + A_f \sin(2 \pi f (t - \phi_f))
```

- This avoids the need to “zero-center” the light-curve, particularly important for incomplete (e.g. flux-limited) list curve.

$(LocalResource("Pics/floatingmean.png"))

-  The previous plot shows a comparison of the standard and floating mean periodograms for data with a frequency of 0.3 and a selection effect which removes faint observations. In this case the mean estimated from the observed data is not close to the true mean, which leads to the failure of the standard periodogram to recover the correct frequency. A floating mean model correctly recovers the true frequency of 0.3.

"""

# ╔═╡ 50a1fb43-f72d-43bd-bea7-99bac76db19e
md"""
- The interpretation of a periodogram as a regular fit procedure opens the way to even more extensions. It is possible, for instance, to model the curve with multiple Fourier components, allowing a greater flexibility:

```math
y_{model}(t;f) = A_f^{0} + \sum_{k=1}^K A_f^{(k)} \sin(2\pi k f (t - \phi_f^{(k)}))
```
"""

# ╔═╡ 8ca6fd17-566d-4aae-961c-cc4a4e6ab1d0
md"""
#### Exercise about multiple component periodgram
***

- We analyse the light-curve of th LINEAR object [14752041](http://simbad.u-strasbg.fr/simbad/sim-id?Ident=linear+14752041&NbIdent=1&Radius=2&Radius.unit=arcmin&submit=submit+id), an eclipsing binary star.
"""

# ╔═╡ 2b0e312c-4285-4e63-9c68-b737646869d9
lnr = DataFrame(CSV.File("LINEAR_14752041.csv"))

# ╔═╡ 171b2e3c-058e-4d82-9d0e-b5dd5f521404
begin
	fg3 = Figure()
	
	ax1fg3 = Axis(fg3[1, 1],
	    xlabel="time (MJD)",
	    ylabel="magnitude",
	    title="LINEAR object 14752041",
	    )
	
	scatter!(lnr[!,:t],lnr[!,:mag])
	errorbars!(lnr[!,:t],lnr[!,:mag],lnr[!,:magerr])
	
	ax1fg3.yreversed=true
	
	fg3
end

# ╔═╡ ec859add-e8e7-413b-b9d4-7d2b8694a46d
md"""
- This looks like a rich light-curve, althgough with a rather irregular sampling and large gaps.

- Before analysing the light-curve let's study the effect of the observing window on the periodogram.
"""

# ╔═╡ eae2ed93-31fb-417a-9621-5586cf4f9732
begin
	ls_window = lombscargle(lnr[!,:t], ones(size(lnr[!,:t])), fit_mean=false, center_data=false, minimum_frequency=0.001, maximum_frequency=10)
	
	
	fg4 = Figure()
	
	ax1fg4 = Axis(fg4[1, 1],
	    xlabel="period (day)",
	    ylabel="power",
	    title="Observing window periodogram",
	    xscale=log10
	    )
	
	lines!(1 ./ ls_window.freq,ls_window.power)
	
	fg4
end

# ╔═╡ f38c7158-5206-467b-987d-38b4aa80e935
md"""
- The window periodogram is indeed quite intersting: togetehr with a rather large peak corresponding to the $\sim 1$ year periodicity of the observations, there is power at several hours and days, reflecting the observing strategy.

- Let's move now to the data periodogram:
"""

# ╔═╡ ce90af7b-ef2c-4cdf-84cb-de76f5dd5c66
begin
	lsper = lombscargle(lnr[!,:t],lnr[!,:mag],lnr[!,:magerr],minimum_frequency=0.001,maximum_frequency=10)
	
	fg5 = Figure(size=(1000,1000))
	
	ax1fg5 = Axis(fg5[1, 1],
	    xlabel="period (day)",
	    ylabel="power",
	    title="LINEAR object 14752041 peeiodogram",
	    xscale=log10
	    )
	
	lines!(1 ./ lsper.freq,lsper.power)
	
	
	fg5
end

# ╔═╡ 0a342241-26b4-425e-b90f-fc555efb0b67
# ╠═╡ show_logs = false
md"""
- The periodogram shows a forest of interesting features, mainly for periods of a few hours.
    - This often happensa when the shape of variability is not close to a sinusoid.
    
- This case can be deal with a multi-term analysis.

$(LocalResource("Pics/higherorder.png"))

- 1-term and 6-term Lomb-Scargle models fit to an eclipsing binary. The standard periodogram finds an alias of the true 17.5-hour frequency, because a simple sinusoidal model cannot closely fit both the primary and secondary eclipse. A six-term Fourier model, on the other hand, does find the true period.

- In principle it is not even necessary to think to sinusoidal terms only. If a more complex signal shape is physically motivated, there are no intrinsic limitations.

- Model comparison is better carried out in a full Bayesian framework, with properly formalized prior knowledge, etc.

- Quite interestingly, it can be shown that the LS periodogram is in fact the optimal statistics for detecting a stationary sinusoidal signal in the presence of Gaussian noise.

"""

# ╔═╡ 17444051-6ee8-44af-8cfc-275bcb495d86
# ╠═╡ show_logs = false
md"""
### Bayesian LS periodogram
***

- The least squares view of the Lomb-Scargle periodogram creates a natural bridge, via maximum likelihood, to Bayesian periodic analysis. In fact, in the Bayesian view, the Lomb-Scargle periodogram is the optimal statistic for detecting a stationary sinusoidal signal in the presence of Gaussian noise.

- For the standard, simple-sinusoid model, the Bayesian periodogram is given by the posterior probability of frequency ``f`` given the data ``D`` and sinusoidal model ``M``:

```math
p(f\mid D, M) \propto e^{P_{LS}(f)}
```

- where ``P_{LS}(f)`` is the LS power. The effect of this exponentiation is to suppress side-lobes and alias peaks in relation to the largest one of the spectrum.

$(LocalResource("Pics/bayesianls.png"))

- Be aware that ``p(f\mid D,M)`` is the probability data are drawn from a sinusoidal model. Not the probability that data are periodic in general.

    - More technically, this is indeed the probability conditioned on the assumption that the data are drawn from a sinusoidal model.

- Some more detail about the derivation of the LS periodogram in a Bayesian framework can be found [here](./open?path=Lectures/Lecture - Lomb-Scargle/Lecture-BayesLS.jl). 
"""

# ╔═╡ 467e6316-fa6d-4487-86cc-8ba408577fe7
# ╠═╡ show_logs = false
cm"""
### LS frequency grid
***

- At variance with the DFT, the frequency grid for LS analysis is not determined by the sampling.

- The most important warning is to pay attention not to choose a too sparse grid, with the risk to miss important features.

$(LocalResource("Pics/lsgrid.png"))

- In general, for an observation length ``T``, we have sinc-shaped peaks of width ``\sim 1/T``. A rule of the thumb advise is typically to oversample the grid by a factor ``\sim 5``.

"""

# ╔═╡ 910a90a8-7b85-4d69-ba8d-3cabbfe89bc4
md"""
### The Window Function
***

- It is clear that a careful evaluation of the window function is crucial in interpreting the results of a LS analysis.

- The LS periodogram can offer a sufficiently reliable way to compute it:

```math
\mathcal{P}_W(f;\{t_n\}) = \left|\sum_{n=1}^{N} e^{-2\pi i f t_n}\right|^2
```

- It is, in its essence, the periodogram for data $g_n=1$ at all times $t_n$. No floating-mean model should be used in this case.

"""

# ╔═╡ bd68480e-e2f6-4759-8333-066001c2ae19
md"""
### LS (or DFT) peak uncertainty
***

- It happens frequenyly thay period uncertainties are quoted basing on the peak width. This is somehow ill-defined, since the width of a periodogram peak depends, essentialy, on the length of the observation.
"""

# ╔═╡ ec7fa3e8-2635-4af3-b2e7-6435e96ebcb2
md"""
#### Exercize about LS (and DFT) periodogram peak uncertainty
***

- First, let's define a function to generate light-curves of different lengths and S/N.
"""

# ╔═╡ 0dafbc5d-d383-44a8-8dc8-704f8a36f7fe
function create_data(N; T=4, signal_to_noise=5, period=1.0, random_state=None)
    rng = Random.seed!(random_state)
    t = T .* rand(N)
    dy = 0.5 ./ signal_to_noise .* ones(size(t))
    y = sin.(2 * π * t ./ period) .+ dy .* randn(N)
    return t, y, dy
end

# ╔═╡ fe1aa970-abc8-4720-9f09-5fe931100263
md"""
- And plot several light-curves with different parameters.
"""

# ╔═╡ 9d413d79-c9fb-4c30-ac02-92380510b607
# ╠═╡ show_logs = false
begin
	fg6 = Figure(size=(2000,1000))
	
	ax1fg6 = Axis(fg6[1, 1],
	    title="Peak scaling with number of data points (fixed S/N=10)",
	    ylabel=L"$P_{LS}$ (normalized)"
	    )
	
	ax2fg6 = Axis(fg6[2, 1],
	    ylabel=L"$P_{LS} / (S/N)^2$ (PSD-normalized)",
	    xlabel="frequency"
	    )
	
	ax3fg6 = Axis(fg6[1, 2],
	    title="Peak scaling with signal-to-noise ratio (fixed N=1000)",
	    ylabel=L"$P_{LS}$ (normalized)"
	    )
	
	ax4fg6 = Axis(fg6[2, 2],
	    ylabel=L"$P_{LS} / (S/N)^2$ (PSD-normalized)",
	    xlabel="frequency"
	    )
	
	
	SN = 10
	for N in [1000, 100, 10]
	    t, y, dy = create_data(N, signal_to_noise=SN, random_state=68345)
	    freq = LinRange(0.01, 4, 2000)
	
	    ls1 = lombscargle(t, y, dy, normalization=:standard, frequencies=freq)
	    ls2 = lombscargle(t, y, dy, normalization=:psd, frequencies=freq)
	
	    lines!(ax1fg6,freq,ls1.power,label="N="*string(N))
	    lines!(ax2fg6,freq,ls2.power ./ SN, label="N="*string(N))
	end
	
	
	N = 1000
	for SN in [10, 1, 0.1]
	    t, y, dy = create_data(N, signal_to_noise=SN, random_state=68345)
	    freq = LinRange(0.01, 4, 2000)
	
	    ls3 = lombscargle(t, y, dy, normalization=:standard, frequencies=freq)
	    ls4 = lombscargle(t, y, dy, normalization=:psd, frequencies=freq)
	
	    lines!(ax3fg6,freq,ls3.power,label="S/N="*string(SN))
	    lines!(ax4fg6,freq,ls4.power ./ SN^2, label="S/N="*string(SN))
	end
	
	axislegend(ax1fg6)
	axislegend(ax2fg6)
	axislegend(ax3fg6)
	axislegend(ax4fg6)
	
	
	
	fg6
end

# ╔═╡ 26d6e75f-dd44-4876-ad72-e5606d27798f
md"""
- The effect of the number of points $N$ and the signal-to-noise ratio $S/N$ on the expected width and height of the periodogram peak.
    - Top panels show the normalized periodogram, while bottom panels show the PSD-normalized periodogram scaled by noise variance. Perhaps surprisingly, neither the number of points nor the signal-to-noise ratio affects the peak width.

- Within the Bayesian interpretation, one can derive an approximate expression relating the uncertainty to the number of samples, $N$, and the average $S/N$ ratio, $Σ$:

```math
\sigma_f \approx f_{1/2} \sqrt{\frac{2}{N\Sigma^2}}
```

"""

# ╔═╡ 66da0ee4-4d50-47e6-ab43-3385b2a9f9fa
# ╠═╡ show_logs = false
md"""
### LS Peak Significance
***

- The typical approach to quantifying the significance of a peak is the *False Alarm Probability* (FAP), which measures the probability that a dataset with no signal would, due to coincidental alignment among the random
errors, lead to a peak of a similar magnitude.

- Even for the LS periodogram, and pure Gaussian noise, it can be proved that the values of the unnormalized periodogram follow a $\chi^2$ distribution with two degrees of freedom.

- If $Z=P(f_0)$ is the periodogram value at a peak with frequency $f_0$, then the cumulative probability to observe a value less then $Z$ is:

```math
P_{single}(Z) = 1 - \exp(-Z)
```

- Analogously to the DFT, we are generally not interested in the distribution of one particular randomly chosen frequency, but rather the distribution of the highest peak of the periodogram.

- The problem here, at variance with the DFT case, is not simple to solve since the value at one frequency is correlated with the value at other frequencies in a way that is quite difficult to analytically express. These correlations come from the convolution with the survey window.

- One common approach is to assume that it can be modeled on some *effective number* of independent frequencies $_{Neff}$, so that the FAP can be estimated as:

```math
FAP(z) \approx 1 - \big[P_{single}(z)\big]^{N_{eff}}
```

- A very simple estimate for $_{Neff}$ can based on the arguments about the expected peak width, $δf = 1/T$. In this approximation, the number of independent peaks in a range $0 ≤ f ≤ f_{max}$ is assumed to be $N_{eff} = f_{max} T$.

- Finding a reliable way to accurately compute $_{Neff}$ analytically is still and open problem. Often the problem is solved computationally by, e.g., a bootstrap procedure.

$(LocalResource("Pics/independentfreqs.png"))

- In the previous plot a comparison of various approaches to computing the FAP for simulated observations with both structured and unstructured survey windows is shown.
"""

# ╔═╡ e30cc0a2-ce7f-4e9f-9f3f-ebc14a76e936
md"""
#### Exercize about peak periodogram significance evaluation
***

- Let's generate a short time-series with the function defined above and compute the sigificance level by means of a boostrap analysis.
"""

# ╔═╡ 2ef299bf-e198-4bde-83cd-ecd86e566683
begin
	t7, y7, dy7 = create_data(30, signal_to_noise=2, random_state=583)
	
	lsper7 = lombscargle(t7, y7, dy7, maximum_frequency=4, samples_per_peak=10)
	
	lsboot7 = LombScargle.bootstrap(1000, t7, y7, dy7, maximum_frequency=4, samples_per_peak=10)
	
	p85 = LombScargle.fapinv(lsboot7,0.85)
	p95 = LombScargle.fapinv(lsboot7,0.95)
	p99 = LombScargle.fapinv(lsboot7,0.99)
	
	
	fg7 = Figure(size=(1000,400))
	
	ax1fg7 = Axis(fg7[1, 1],
	    )
	
	ax2fg7 = Axis(fg7[1, 2],
	    )
	
	scatter!(ax1fg7,t7,y7)
	errorbars!(ax1fg7,t7,y7,dy7)
	
	lines!(ax2fg7,lsper7.freq,lsper7.power)
	hlines!(ax2fg7,p85,label="85%")
	hlines!(ax2fg7,p95,label="95%")
	hlines!(ax2fg7,p99,label="99%")
	
	axislegend(ax2fg7)
	
	fg7
end

# ╔═╡ 6d2ccb03-4455-4249-9d3d-be4cb5d7a45f
md"""
## Various possible PSD normalizations
***

- When considering the periodogram from the Fourier perspective, it is useful to normalize the periodogram such that in the special case of equally spaced data it recovers the standard Fourier power spectrum.

- This is the so-called "psd" normalization, and the equivalent least-squares expression is:

```math
P(f) = \frac{1}{2}\big[\hat{\chi}^2_0 - \hat{\chi}^2(f)\big]
```

- For equally spaced data this becomes:

```math
P(f) = \frac{1}{N} \left| FFT(y_n) \right|^2
```

- Under the "psd" normalization periodogram units are “unit(y)$^2$”. And can be interpreted as squared amplitudes of the Fourier component at each frequency.

- However, if uncertainties are included in the analysis, the periodogram becomes unitless: it is essentially a measure of periodic content in signal-to-noise ratio rather than in signal itself.

- In the least-squares view of the periodogram, the periodogram is interpreted as an inverse measure of the goodness of  fit for a model.
    - If the sinusoidal model perfectly fits the data at some frequency $f_0$, then $\hat{\chi}^2(f_0) = 0$ and the periodogram is maximized at a value of $\hat{\chi}^2_0 / 2$.
    
- The minimum value of the periodogram can only be 0, and  therefore a possible normalization that keep the (unitless) values between 0 and 1 is:

```math
P_{norm}(f) = 1 - \frac{\hat{\chi}^2(f)}{\hat{\chi}^2_0}
```
"""

# ╔═╡ 78119e66-703b-4305-85cd-956f1053cc60
md"""
## Reference & Material

Material and papers related to the topics discussed in this lecture.

- [Vander Plas (2017) - "Understanding the Lomb-Scargle Periodogram”](https://ui.adsabs.harvard.edu/abs/2018ApJS..236...16V/abstract)
"""

# ╔═╡ 435fd658-c227-47ce-8dd6-cca20bff2a26
md"""
## Further Material

Papers for examining more closely some of the discussed topics.

- [Makarov et al. (2024) - "Robust 1-norm Periodograms for Analysis of Noisy Non-Gaussian Time Series with Irregular Cadences: Application to VLBI Astrometry of Quasars"](https://ui.adsabs.harvard.edu/abs/2024PASP..136e4503M/abstract)
- [Nilsson et al. (2018) - "Long-term optical monitoring of TeV emitting blazars. I. Data analysis"](https://ui.adsabs.harvard.edu/abs/2018A%26A...620A.185N/abstract)
- [Tarnopolski & Marchenko (2021) - "A Comprehensive Power Spectral Density Analysis of Astronomical Time Series. II. The Swift/BAT Long Gamma-Ray Bursts"](https://ui.adsabs.harvard.edu/abs/2021ApJ...911...20T/abstract)
- [Vio et al. (2010) - "Unevenly-sampled signals: a general formalism for the Lomb-Scargle periodogram"](https://ui.adsabs.harvard.edu/abs/2010A%26A...519A..85V/abstract)
- [Glynn et al. (2006) - "Detecting periodic patterns in unevenly spaced gene expression time series using Lomb–Scargle periodograms"](https://academic.oup.com/bioinformatics/article-pdf/22/3/310/48839224/bioinformatics_22_3_310.pdf)
- [Emmanoulopoulos et al. (2013) - "Generating artificial light curves: revisited and updated"](https://ui.adsabs.harvard.edu/abs/2013MNRAS.433..907E/abstract)
- [Perig et al. (2019) - "Periodicity in Volcanic Gas Plumes: A Review and Analysis"](https://ui.adsabs.harvard.edu/abs/2019Geosc...9..394P/abstract)
"""

# ╔═╡ 427b330c-e92a-4be3-8db2-4373f91f4d0e
md"""
### Credits
***

This notebook contains material obtained from [https://github.com/jakevdp/PracticalLombScargle/blob/master/figures/LombScargleVsClassical.ipynb](https://github.com/jakevdp/PracticalLombScargle/blob/master/figures/LombScargleVsClassical.ipynb), [https://github.com/jakevdp/PracticalLombScargle/blob/master/figures/Uncertainty.ipynb](https://github.com/jakevdp/PracticalLombScargle/blob/master/figures/Uncertainty.ipynb) and from [https://github.com/jakevdp/PracticalLombScargle/blob/master/figures/LINEAR_binary.ipynb](https://github.com/jakevdp/PracticalLombScargle/blob/master/figures/LINEAR_binary.ipynb).
"""

# ╔═╡ 6d25216d-ab50-4ce2-993b-84ce3070eb4e
cm"""
## Course Flow

<table>
  <tr>
    <td>Previous lecture</td>
    <td>Next lecture</td>
  </tr>
  <tr>
    <td><a href="./open?path=Lectures/Science Case - X-Ray Binaries/Lecture-X-RayBinaries.jl">Science case: X-ray binariess</a></td>
    <td><a href="./open?path=Lectures/Science Case - Variable Stars/Lecture-VariableStars.jl">Science case about variable stars</a></td>
  </tr>
 </table>


"""

# ╔═╡ aa791077-751f-4542-8dc3-6cbb392c2302
md"""
**Copyright**

This notebook is provided as [Open Educational Resource](https://en.wikipedia.org/wiki/Open_educational_resources). Feel free to use the notebook for your own purposes. The text is licensed under [Creative Commons Attribution 4.0](https://creativecommons.org/licenses/by/4.0/), the code of the examples, unless obtained from other properly quoted sources, under the [MIT license](https://opensource.org/licenses/MIT). Please attribute the work as follows: *Stefano Covino, Time Domain Astrophysics - Lecture notes featuring computational examples, 2025*.
"""

# ╔═╡ Cell order:
# ╟─ff837227-c0bc-433c-b462-9e18ff3d947b
# ╟─53a01399-1f0b-4fbb-9966-97124e9eff42
# ╠═4bae382c-eb38-4b21-8b61-67d238c80d19
# ╠═414e8cd2-d282-4920-b4e8-91ed552467cf
# ╠═93f1b40e-8990-405d-bba1-9f98abd054ca
# ╟─f7100fe5-2cfd-448d-abcc-66a623a8f11f
# ╟─31dd4ceb-8afd-4b55-aacb-2d888681ad1c
# ╟─e6386bbc-ea18-49a6-b64a-51b627d31565
# ╟─477704cc-f022-4cc5-a4a4-679387b819bd
# ╟─f95ec17d-9c56-4897-b5b2-65c3470fc294
# ╟─0f01200a-665e-4ae2-a6b9-b5c192955f3f
# ╟─a24629ad-7302-4aff-b7bf-fcc7635b9691
# ╟─533e41e2-08c3-450b-83eb-ae19664a6cfe
# ╟─ae7829b7-b652-4e54-9231-f7bc3c3b6093
# ╟─9a422815-783a-473c-8538-6ecf5c208bb0
# ╟─6fa31a81-cbbe-401b-acb0-0d44b0980363
# ╟─367c08ff-cedd-4e63-b53a-3178b0ec45cb
# ╠═ef919d43-ffb2-404f-94e4-a10b9a543e09
# ╟─ce5b3710-ac14-44f5-85d8-abc9756c1f11
# ╠═18958f04-3eb0-4e4e-9957-ca0c5540e5a5
# ╟─b985cda2-8242-468b-b5de-2299bfa50b33
# ╠═2e3e0747-48c2-43d1-abd4-b15f61a402bd
# ╠═ccb2f0e6-517b-4860-bac2-dac964e8d388
# ╠═0b098634-7890-4551-8343-34dbba3159ba
# ╟─2503892b-3bcf-41b8-a636-562d2613bd59
# ╠═f652aeb7-bcf0-481c-924c-412f406b3542
# ╟─637108d6-c977-4a2d-b312-e40bf375d5f1
# ╟─325a6b84-af05-4d78-94a2-3674a501c1c2
# ╟─50a1fb43-f72d-43bd-bea7-99bac76db19e
# ╟─8ca6fd17-566d-4aae-961c-cc4a4e6ab1d0
# ╠═2b0e312c-4285-4e63-9c68-b737646869d9
# ╠═171b2e3c-058e-4d82-9d0e-b5dd5f521404
# ╟─ec859add-e8e7-413b-b9d4-7d2b8694a46d
# ╠═eae2ed93-31fb-417a-9621-5586cf4f9732
# ╟─f38c7158-5206-467b-987d-38b4aa80e935
# ╠═ce90af7b-ef2c-4cdf-84cb-de76f5dd5c66
# ╟─0a342241-26b4-425e-b90f-fc555efb0b67
# ╟─17444051-6ee8-44af-8cfc-275bcb495d86
# ╟─467e6316-fa6d-4487-86cc-8ba408577fe7
# ╟─910a90a8-7b85-4d69-ba8d-3cabbfe89bc4
# ╟─bd68480e-e2f6-4759-8333-066001c2ae19
# ╟─ec7fa3e8-2635-4af3-b2e7-6435e96ebcb2
# ╠═0dafbc5d-d383-44a8-8dc8-704f8a36f7fe
# ╟─fe1aa970-abc8-4720-9f09-5fe931100263
# ╠═9d413d79-c9fb-4c30-ac02-92380510b607
# ╟─26d6e75f-dd44-4876-ad72-e5606d27798f
# ╟─66da0ee4-4d50-47e6-ab43-3385b2a9f9fa
# ╟─e30cc0a2-ce7f-4e9f-9f3f-ebc14a76e936
# ╠═2ef299bf-e198-4bde-83cd-ecd86e566683
# ╟─6d2ccb03-4455-4249-9d3d-be4cb5d7a45f
# ╟─78119e66-703b-4305-85cd-956f1053cc60
# ╟─435fd658-c227-47ce-8dd6-cca20bff2a26
# ╟─427b330c-e92a-4be3-8db2-4373f91f4d0e
# ╟─6d25216d-ab50-4ce2-993b-84ce3070eb4e
# ╟─aa791077-751f-4542-8dc3-6cbb392c2302
